"use strict";
/**
 * This module offers a function to load shared library fullfilling the [[SharedLibApi]] contract.
 *
 * @module @wontok/node-addon-shared-lib-loader
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.initSharedLib = void 0;
/** Implementation */
var node_addon_shared_lib_native_api_1 = require("./node-addon-shared-lib-native-api");
var utils_1 = require("./utils");
/**
 * Loads and intializez a shared library. This method should be called exacly once per shared library!
 *
 * @param sharedLib the details of the shared library to be loaded.
 * @param initOptions a plain object - will be sent to the initSymbol of the library.
 * @param callbacks an array of callbacks which can handle events from the shared library. If an event starts with `development` it will be ignored. If an event cannot be found an error will be thrown.
 * @returns a function which can be used to send commands to the shared library.
 */
var initSharedLib = function (sharedLib, initOptions, callbacks) {
    var callbacksMap = Object.fromEntries(callbacks.map(function (callback) { return [callback.callbackName, callback]; }));
    var isUnloaded = false;
    var sharedLibId = node_addon_shared_lib_native_api_1.Addon.init(sharedLib.path, sharedLib.initSymbol, sharedLib.executeCommandsSymbol, JSON.stringify(initOptions), function (callbackParamsAsJson) {
        utils_1.debug("Received raw event text in JS Land: " + callbackParamsAsJson + "\n");
        try {
            var callbackParams = JSON.parse(callbackParamsAsJson);
            var callbackName = callbackParams.callbackName;
            if (undefined !== callbackName &&
                callbackName.startsWith("development")) {
                return;
            }
            var callback = callbacksMap[callbackParams.callbackName];
            if (undefined === callback) {
                throw new Error("Undefined callback: " + callbackParams.callbackName);
            }
            try {
                callback(callbackParams);
                return;
            }
            catch (error) {
                // TODO: So what to do?
                utils_1.debug("Erroneous event handling !: " + error + "\n");
            }
        }
        catch (error) {
            // TODO: So what to do?
            utils_1.debug("Malfolrmed event!: " + error + "\n");
        }
    }, function () {
        isUnloaded = true;
    });
    return {
        executeCommands: function (commands) {
            var commandsAsJson = JSON.stringify(commands);
            utils_1.debug("Sending commands to native addon: " + commandsAsJson + "\n");
            node_addon_shared_lib_native_api_1.Addon.executeCommands(sharedLibId, commandsAsJson);
        },
        unload: function () {
            node_addon_shared_lib_native_api_1.Addon.unload(sharedLibId);
            return new Promise(function (resolve, _) {
                var intervalId = null;
                intervalId = setInterval(function () {
                    if (isUnloaded) {
                        clearInterval(intervalId);
                        resolve();
                    }
                }, 100);
            });
        },
    };
};
exports.initSharedLib = initSharedLib;
